import {
	createPortal,
	useState,
	useRef,
	Fragment,
	createElement,
} from '@wordpress/element'
import classnames from 'classnames'
import { CheckboxControl } from '@wordpress/components'

import { __ } from 'ct-i18n'

import { getSettings } from '@wordpress/date'

import { TimePicker } from '@wordpress/components'

import { usePopoverMaker, OutsideClickHandler } from 'noorgfx-options'

const humanReadableDays = (day) =>
	({
		monday: __('Mon', 'noorgfx-component'),
		tuesday: __('Tue', 'noorgfx-component'),
		wednesday: __('Wed', 'noorgfx-component'),
		thursday: __('Thu', 'noorgfx-component'),
		friday: __('Fri', 'noorgfx-component'),
		saturday: __('Sat', 'noorgfx-component'),
		sunday: __('Sun', 'noorgfx-component'),
	}[day])

const ScheduleDate = ({ onChange, condition }) => {
	const inputRef = useRef()
	const [isOpen, setIsOpen] = useState(false)

	const { styles, popoverProps } = usePopoverMaker({
		ref: inputRef,
		defaultHeight: 228,
		shouldCalculate: true,
	})

	const defaultValue = {
		monday: true,
		tuesday: true,
		wednesday: true,
		thursday: true,
		friday: true,
		saturday: true,
		sunday: true,
	}

	const activeDays = Object.keys(defaultValue).filter(
		(key) => (condition.payload.days || defaultValue)[key]
	)

	let preview =
		'Only ' + activeDays.map((day) => humanReadableDays(day)).join(', ')

	if (activeDays.length === 7) {
		preview = __('Every day', 'noorgfx-component')
	}

	if (
		activeDays.length === 2 &&
		activeDays.includes('saturday') &&
		activeDays.includes('sunday')
	) {
		preview = __('Only weekends', 'noorgfx-component')
	}

	if (
		activeDays.length === 5 &&
		!activeDays.includes('saturday') &&
		!activeDays.includes('sunday')
	) {
		preview = __('Only weekdays', 'noorgfx-component')
	}

	if (activeDays.length === 0) {
		preview = __('Never', 'noorgfx-component')
	}

	const defaultStart = condition.payload.time_start || '00:00'
	const defaultEnd = condition.payload.time_end || '23:59'

	const startDate = new Date()

	startDate.setHours(defaultStart.split(':')[0])
	startDate.setMinutes(defaultStart.split(':')[1], 0, 0)

	const endDate = new Date()

	endDate.setHours(defaultEnd.split(':')[0])
	endDate.setMinutes(defaultEnd.split(':')[1], 0, 0)

	const settings = getSettings()

	// To know if the current timezone is a 12 hour time with look for "a" in the time format
	// We also make sure this a is not escaped by a "/"
	const is12HourTime = /a(?!\\)/i.test(
		settings.formats.time
			.toLowerCase() // Test only the lower case a.
			.replace(/\\\\/g, '') // Replace "//" with empty strings.
			.split('')
			.reverse()
			.join('') // Reverse the string and test for "a" not followed by a slash.
	)

	return (
		<Fragment>
			<OutsideClickHandler
				className="ct-select-input"
				wrapperProps={{
					ref: inputRef,
					role: 'combobox',
					'aria-expanded': isOpen,
					'aria-haspopup': 'listbox',
					onClick: () => {
						setIsOpen(!isOpen)
					},
				}}
				onOutsideClick={(e) => {
					if (e.target.closest('.ct-select-dropdown')) {
						return
					}

					setIsOpen(false)
				}}>
				<input value={preview} onChange={() => {}} readOnly />
			</OutsideClickHandler>

			{isOpen &&
				createPortal(
					<div
						className={classnames(
							'ct-select-dropdown ct-recurring-scheduling-dropdown',
							{
								'ct-fixed': true,
							}
						)}
						{...popoverProps}
						style={styles}>
						<div className="ct-recurring-scheduling-days">
							<label className="ct-label">
								{__('Recurring Days', 'noorgfx-component')}
							</label>

							{[
								{
									key: 'monday',
									value: __('Monday', 'noorgfx-component'),
								},

								{
									key: 'tuesday',
									value: __('Tuesday', 'noorgfx-component'),
								},

								{
									key: 'wednesday',
									value: __('Wednesday', 'noorgfx-component'),
								},

								{
									key: 'thursday',
									value: __('Thursday', 'noorgfx-component'),
								},

								{
									key: 'friday',
									value: __('Friday', 'noorgfx-component'),
								},

								{
									key: 'saturday',
									value: __('Saturday', 'noorgfx-component'),
								},

								{
									key: 'sunday',
									value: __('Sunday', 'noorgfx-component'),
								},
							].map(({ key, value }) => (
								<CheckboxControl
									label={value}
									checked={
										(condition.payload.days ||
											defaultValue)[key]
									}
									onChange={() => {
										onChange({
											...condition,
											payload: {
												...condition.payload,
												days: {
													...(condition.payload
														.days || defaultValue),
													[key]: !(condition.payload
														.days || defaultValue)[
														key
													],
												},
											},
										})
									}}
								/>
							))}
						</div>

						<div className="ct-recurring-scheduling-time">
							<div className="ct-recurring-start-time">
								<label className="ct-label">
									{__('Start Time', 'noorgfx-component')}
								</label>

								<TimePicker
									is12Hour={is12HourTime}
									currentTime={startDate}
									onChange={(time) => {
										onChange({
											...condition,
											payload: {
												...condition.payload,
												time_start: wp.date.format(
													'H:i',
													time
												),
											},
										})
									}}
								/>
							</div>

							<div className="ct-recurring-stop-time">
								<label className="ct-label">
									{__('Stop Time', 'noorgfx-component')}
								</label>

								<TimePicker
									is12Hour={is12HourTime}
									currentTime={endDate}
									onChange={(time) => {
										onChange({
											...condition,
											payload: {
												...condition.payload,
												time_end: wp.date.format(
													'H:i',
													time
												),
											},
										})
									}}
								/>
							</div>
						</div>
					</div>,

					document.body
				)}
		</Fragment>
	)
}

export default ScheduleDate
