jQuery(document).ready(function($) {
    let hasServiceField = $('#sab-service').length > 0;

    function generateJalaaliMonthCalendar(jYear, jMonth, availableDates) {
        const daysInMonth = jalaali.jalaaliMonthLength(jYear, jMonth + 1);
        const gregorianStart = jalaali.toGregorian(jYear, jMonth + 1, 1);
        const gregorianDateObj = new Date(gregorianStart.gy, gregorianStart.gm - 1, gregorianStart.gd);
        let firstDayOfMonth = gregorianDateObj.getDay();
        firstDayOfMonth = (firstDayOfMonth + 1) % 7;

        let calendarHtml = `<div class="calendar-month">`;
        calendarHtml += `<div class="month-header">${getJalaaliMonthName(jMonth)} ${jYear}</div>`;
        calendarHtml += `<div class="calendar-days">`;

        const jalaaliDaysFull = ["شنبه", "یکشنبه", "دوشنبه", "سه شنبه", "چهارشنبه", "پنج شنبه", "جمعه"];
        let jalaaliDays = window.innerWidth < 768 ? jalaaliDaysFull.map(day => day.charAt(0)) : jalaaliDaysFull;

        jalaaliDays.forEach(day => {
            calendarHtml += `<div class="day-header">${day}</div>`;
        });

        for (let i = 0; i < firstDayOfMonth; i++) {
            calendarHtml += `<div class="calendar-day empty"></div>`;
        }

        for (let jDay = 1; jDay <= daysInMonth; jDay++) {
            const gregorianDate = jalaali.toGregorian(jYear, jMonth + 1, jDay);
            const dateString = `${gregorianDate.gy}-${gregorianDate.gm.toString().padStart(2, '0')}-${gregorianDate.gd.toString().padStart(2, '0')}`;
            const isAvailable = availableDates.includes(dateString);

            let className = 'calendar-day';
            className += isAvailable ? ' available' : ' unavailable';
            
            calendarHtml += `<div class="${className}" data-date="${dateString}">${jDay}</div>`;
        }

        calendarHtml += `</div></div>`;
        return calendarHtml;
    }

    function getJalaaliMonthName(month) {
        const months = ["فروردین", "اردیبهشت", "خرداد", "تیر", "مرداد", "شهریور", "مهر", "آبان", "آذر", "دی", "بهمن", "اسفند"];
        return months[month];
    }

    function getDayName(dateString) {
        const days = ["شنبه", "یکشنبه", "دوشنبه", "سه شنبه", "چهارشنبه", "پنج شنبه", "جمعه"];
        const date = new Date(dateString);
        let dayIndex = date.getDay();
        dayIndex = (dayIndex + 1) % 7;
        return days[dayIndex];
    }

    function convertToJalaali(dateString) {
        const dateParts = dateString.split("-");
        const gregorianYear = parseInt(dateParts[0], 10);
        const gregorianMonth = parseInt(dateParts[1], 10);
        const gregorianDay = parseInt(dateParts[2], 10);
        const jalaaliDate = jalaali.toJalaali(gregorianYear, gregorianMonth, gregorianDay);
        const month = jalaaliDate.jm.toString().padStart(2, '0');
        const day = jalaaliDate.jd.toString().padStart(2, '0');
        return `${jalaaliDate.jy}/${month}/${day}`;
    }

    function isDatePassed(dateString) {
        const today = new Date();
        today.setHours(0, 0, 0, 0);
        const checkDate = new Date(dateString);
        checkDate.setHours(0, 0, 0, 0);
        return checkDate < today;
    }

    function showLoading() {
        Swal.fire({
            title: "لطفاً صبر کنید...",
            text: "در حال بارگذاری اطلاعات",
            allowOutsideClick: false,
            didOpen: () => {
                Swal.showLoading();
            },
        });
    }

    function hideLoading() {
        Swal.close();
    }

	function loadDoctors(serviceId = 0, showLoading = false) {
		var $doctorSelect = $("#sab-doctor");
		$doctorSelect.prop("disabled", true).empty().append('<option value="">لطفا انتخاب کنید</option>');

		if (showLoading) {
			Swal.fire({
				title: "لطفاً صبر کنید...",
				text: "در حال بارگذاری اطلاعات",
				allowOutsideClick: false,
				didOpen: () => {
					Swal.showLoading();
				},
			});
		}

		$.ajax({
			url: sab_ajax_object.ajax_url,
			method: "POST",
			data: { action: "sab_get_doctors", service_id: serviceId },
			success: function (response) {
				if (response.success) {
					var doctors = response.data;
					$.each(doctors, function (index, doctor) {
						$doctorSelect.append('<option value="' + doctor.id + '">' + doctor.name + "</option>");
					});
					$doctorSelect.prop("disabled", false);
				} else {
					Swal.fire("خطا", "هیچ سرویس دهنده ای یافت نشد.", "error");
				}
			},
			error: function () {
				Swal.fire("خطا", "خطا در بارگذاری سرویس دهندگان.", "error");
			},
			complete: function () {
				if (showLoading) {
					Swal.close();
				}
			}
		});
	}

	if (hasServiceField) {
		$.ajax({
			url: sab_ajax_object.ajax_url,
			method: "POST",
			data: { action: "sab_get_services" },
			success: function (response) {
				if (response.success) {
					var services = response.data;
					var $serviceSelect = $("#sab-service");
					$.each(services, function (index, service) {
						$serviceSelect.append('<option value="' + service.id + '">' + service.name + "</option>");
					});
				}
			},
		});

		$("#sab-service").change(function () {
			var serviceId = $(this).val();
			if (serviceId) {
				loadDoctors(serviceId, true);
			}
		});
	} else {
		loadDoctors(0, false);
	}

    let availableDates = [];
    let availableTimes = {};
    let bookedAppointments = [];

    $("#sab-doctor").change(function () {
        var doctorId = $(this).val();
        var $dateButton = $("#sab-select-date");
        var $timeButton = $("#sab-select-time");
        var $dateInput = $("#sab-days");
        var $timeInput = $("#sab-times");

        $dateButton.prop("disabled", true).text("انتخاب تاریخ");
        $timeButton.prop("disabled", true).text("انتخاب زمان");
        $dateInput.val("");
        $timeInput.val("");
        availableDates = [];
        availableTimes = {};
        bookedAppointments = [];

        if (doctorId) {
            showLoading();
            $.ajax({
                url: sab_ajax_object.ajax_url,
                method: "POST",
                data: { action: "sab_get_availability", doctor_id: doctorId },
                success: function (response) {
                    if (response.success) {
                        var dates = response.data.dates;
                        availableTimes = response.data.times;

                        dates.sort(function (a, b) {
                            return new Date(a) - new Date(b);
                        });

                        availableDates = dates.filter((date) => !isDatePassed(date));

                        if (availableDates.length > 0) {
                            $dateButton.prop("disabled", false);
                        }

                        $.ajax({
                            url: sab_ajax_object.ajax_url,
                            method: "POST",
                            data: {
                                action: "sab_get_booked_appointments",
                                doctor_id: doctorId,
                            },
                            success: function (bookedResponse) {
                                if (bookedResponse.success) {
                                    bookedAppointments = bookedResponse.data;
                                }
                            },
                        });
                    }
                },
                error: function () {
                    Swal.fire("خطا", "خطا در بارگذاری تاریخ ها.", "error");
                },
                complete: function () {
                    hideLoading();
                },
            });
        }
    });

    $("#sab-select-date").click(function () {
        if (availableDates.length === 0) {
            Swal.fire({
                title: "خطا",
                text: "هیچ نوبت آزادی در دسترس نیست.",
                icon: "error",
                confirmButtonText: "باشه"
            });
            return;
        }

        const months = {};
        availableDates.forEach(date => {
            const [gYear, gMonth, gDay] = date.split('-').map(Number);
            const jalaaliDate = jalaali.toJalaali(gYear, gMonth, gDay);
            const key = `${jalaaliDate.jy}-${jalaaliDate.jm}`;
            if (!months[key]) {
                months[key] = { jYear: jalaaliDate.jy, jMonth: jalaaliDate.jm - 1 };
            }
        });

        let calendarHtml = `
            <div class="swiper-container">
                <div class="swiper-wrapper">
        `;

        Object.values(months).forEach(({ jYear, jMonth }) => {
            calendarHtml += `
                <div class="swiper-slide">
                    ${generateJalaaliMonthCalendar(jYear, jMonth, availableDates)}
                </div>
            `;
        });

        calendarHtml += `
                </div>
                <div class="swiper-button-next"></div>
                <div class="swiper-button-prev"></div>
            </div>
        `;

        Swal.fire({
            title: "انتخاب تاریخ",
            html: calendarHtml,
            confirmButtonText: "تایید",
            cancelButtonText: "انصراف",
            showCancelButton: true,
            customClass: {
                popup: "swal-rtl",
            },
            width: '600px',
            didOpen: () => {
                const swiper = new Swiper('.swiper-container', {
                    slidesPerView: 1,
                    spaceBetween: 0,
                    navigation: {
                        nextEl: '.swiper-button-next',
                        prevEl: '.swiper-button-prev',
                    },
                    autoHeight: true,
                    watchOverflow: true,
                });

                $('.calendar-day.available').on('click', function () {
                    $('.calendar-day').removeClass('selected');
                    $(this).addClass('selected');
                });
            },
            preConfirm: () => {
                const selectedDate = $('.calendar-day.selected').data('date');
                if (!selectedDate) {
                    Swal.showValidationMessage("لطفاً یک تاریخ را انتخاب کنید");
                    return false;
                }
                return selectedDate;
            },
        }).then((result) => {
            if (result.isConfirmed) {
                const selectedDate = result.value;
                $("#sab-days").val(selectedDate);
                const jalaaliDate = convertToJalaali(selectedDate);
                const dayName = getDayName(selectedDate);
                $("#sab-select-date").text(`${dayName} ${jalaaliDate}`);
                $("#sab-select-time").prop("disabled", false).text("انتخاب زمان");
                $("#sab-times").val("");
            }
        });
    });

    function isTimeSlotPassed(selectedDate, selectedTime) {
        const currentDate = new Date();
        const [currentHour, currentMinute] = [currentDate.getHours(), currentDate.getMinutes()];
        const [slotHour, slotMinute] = selectedTime.split(':').map(Number);
        const currentTimeInMinutes = currentHour * 60 + currentMinute;
        const slotTimeInMinutes = slotHour * 60 + slotMinute;
        return (
            new Date(selectedDate).toDateString() === currentDate.toDateString() &&
            currentTimeInMinutes >= slotTimeInMinutes
        );
    }

    $("#sab-select-time").click(function () {
        const selectedDate = $("#sab-days").val();
        if (!selectedDate || !availableTimes[selectedDate]) return;

        let timeHtml = `<div class="sab-time-radio-group">`;
        availableTimes[selectedDate].forEach((time) => {
            const trimmedTime = time.trim();
            const isBooked = bookedAppointments.some(
                (appointment) => appointment.day === selectedDate && appointment.time === trimmedTime
            );
            const isTimePassed = isTimeSlotPassed(selectedDate, trimmedTime);

            timeHtml += `
                <label class="${isBooked || isTimePassed ? "label-booked" : "label-available"}">
                    <input type="radio" name="popup-time" value="${trimmedTime}" ${isBooked || isTimePassed ? "disabled" : ""}>
                    <span>${trimmedTime}</span>
                </label>`;
        });
        timeHtml += "</div>";

        Swal.fire({
            title: "انتخاب زمان",
            html: timeHtml,
            confirmButtonText: "تایید",
            cancelButtonText: "انصراف",
            showCancelButton: true,
            customClass: {
                popup: "swal-rtl",
            },
            preConfirm: () => {
                const selectedTime = document.querySelector('input[name="popup-time"]:checked');
                if (!selectedTime) {
                    Swal.showValidationMessage("لطفاً یک زمان را انتخاب کنید");
                    return false;
                }
                return selectedTime.value;
            },
        }).then((result) => {
            if (result.isConfirmed) {
                const selectedTime = result.value;
                $("#sab-times").val(selectedTime);
                $("#sab-select-time").text(selectedTime);
            }
        });
    });

	$('#sab-appointment-form').submit(function (e) {
		e.preventDefault();
		var selectedDate = $('#sab-days').val();
		var selectedTime = $('#sab-times').val();
		if (!selectedDate || !selectedTime) {
			Swal.fire({
				title: 'خطا',
				text: 'لطفاً یک روز و ساعت معتبر انتخاب کنید.',
				icon: 'error',
				confirmButtonText: 'بازگشت به فرم',
				customClass: {
					popup: 'swal-rtl'
				}
			});
			return;
		}

		var formData = {
			action: 'sab_book_appointment',
			service_id: $('#sab-service').val() || 0,
			doctor_id: $('#sab-doctor').val(),
			date: selectedDate,
			time: selectedTime,
			name: $('#sab-name').val(),
			phone: $('#sab-phone').val(),
			mobile: $('#sab-mobile').val(),
			custom_text_1: $('#custom_text_1').val(),
			custom_text_2: $('#sab-custom_text_2').val(),
			custom_text_3: $('#sab-custom_text_3').val(),
			custom_text_4: $('#sab-custom_text_4').val(),
			email: $('#sab-email').val()
		};

		var serviceName = $('#sab-service option:selected').text() || "نامشخص";
		var doctorName = $('#sab-doctor option:selected').text();
		var jalaaliDate = convertToJalaali(selectedDate);
		var dayName = getDayName(selectedDate);

		function getLabelByFieldId(fieldId) {
			return $(`label[for="${fieldId}"]`).text() || $(`#${fieldId}`).closest('.sab-field').find('label').text();
		}

		var serviceLabel = getLabelByFieldId('sab-service') || 'خدمات';
		var doctorLabel = getLabelByFieldId('sab-doctor') || 'سرویس دهنده';
		var dateLabel = getLabelByFieldId('date') || 'تاریخ';
		var timeLabel = getLabelByFieldId('time') || 'زمان';
		var nameLabel = getLabelByFieldId('sab-name') || 'نام ارسال کننده';
		var phoneLabel = getLabelByFieldId('sab-phone') || 'تلفن تماس';
		var mobileLabel = getLabelByFieldId('sab-mobile') || 'تلفن همراه';
		var emailLabel = getLabelByFieldId('sab-email') || 'ایمیل';
		var customText1Label = getLabelByFieldId('custom_text_1') || 'فیلد سفارشی ۱';
		var customText2Label = getLabelByFieldId('custom_text_2') || 'فیلد سفارشی ۲';
		var customText3Label = getLabelByFieldId('custom_text_3') || 'فیلد سفارشی ۳';
		var customText4Label = getLabelByFieldId('custom_text_4') || 'فیلد سفارشی ۴';

		var confirmationHtml = `
			<div style="text-align: right; direction: rtl;">
				${formData.name ? `<p><strong>${nameLabel}:</strong> ${formData.name}</p>` : ''}
				${hasServiceField ? `<p><strong>${serviceLabel}:</strong> ${serviceName}</p>` : ''}
				<p><strong>${doctorLabel}:</strong> ${doctorName}</p>
				<p><strong>${dateLabel}:</strong> ${dayName} ${jalaaliDate}</p>
				<p><strong>${timeLabel}:</strong> ${formData.time}</p>
				${formData.phone ? `<p><strong>${phoneLabel}:</strong> ${formData.phone}</p>` : ''}
				${formData.mobile ? `<p><strong>${mobileLabel}:</strong> ${formData.mobile}</p>` : ''}
				${formData.email ? `<p><strong>${emailLabel}:</strong> ${formData.email}</p>` : ''}
				${formData.custom_text_1 ? `<p><strong>${customText1Label}:</strong> ${formData.custom_text_1}</p>` : ''}
				${formData.custom_text_2 ? `<p><strong>${customText2Label}:</strong> ${formData.custom_text_2}</p>` : ''}
				${formData.custom_text_3 ? `<p><strong>${customText3Label}:</strong> ${formData.custom_text_3}</p>` : ''}
				${formData.custom_text_4 ? `<p><strong>${customText4Label}:</strong> ${formData.custom_text_4}</p>` : ''}
			</div>
		`;

		Swal.fire({
			title: 'تأیید اطلاعات رزرو',
			html: confirmationHtml,
			icon: 'info',
			showCancelButton: true,
			confirmButtonText: 'تأیید و رزرو نوبت',
			cancelButtonText: 'انصراف',
			customClass: {
				popup: 'swal-rtl'
			}
		}).then((result) => {
			if (result.isConfirmed) {
				Swal.fire({
					title: 'درحال بررسی',
					text: 'لطفاً منتظر بمانید...',
					allowOutsideClick: false,
					didOpen: () => {
						Swal.showLoading();
					}
				});

				var formData = new FormData($('#sab-appointment-form')[0]);
				formData.append('action', 'sab_book_appointment');

				$.ajax({
					url: sab_ajax_object.ajax_url,
					method: 'POST',
					data: formData,
					contentType: false,
					processData: false,
					success: function (response) {
						Swal.close();
						if (response.success) {
							Swal.fire({
								title: 'ثبت اطلاعات موفق',
								text: response.data,
								icon: 'success',
								confirmButtonText: 'بازگشت',
								customClass: {
									popup: 'swal-rtl'
								}
							}).then((result) => {
								if (result.isConfirmed) {
									window.location.reload();
								}
							});
						} else {
							Swal.fire({
								title: 'خطا',
								text: response.data,
								icon: 'error',
								confirmButtonText: 'بازگشت',
								customClass: {
									popup: 'swal-rtl'
								}
							});
						}
					},
					error: function (xhr, status, error) {
						Swal.close();
						Swal.fire({
							title: 'خطا',
							text: 'خطا در ارسال درخواست',
							icon: 'error',
							confirmButtonText: 'بازگشت',
							customClass: {
								popup: 'swal-rtl'
							}
						});
					}
				});
			}
		});
	});

    const canvas = document.getElementById('sab-captcha-canvas');
    if (canvas) {
        const ctx = canvas.getContext('2d');
        canvas.width = 100;
        canvas.height = 50;

        function generateCaptcha() {
            const captcha = Math.floor(1000 + Math.random() * 9000).toString();
            drawCaptcha(captcha);
            return captcha;
        }

        function drawCaptcha(captcha) {
            ctx.clearRect(0, 0, canvas.width, canvas.height);
            ctx.fillStyle = '#f3f3f3';
            ctx.fillRect(0, 0, canvas.width, canvas.height);

            for (let i = 5; i < 10; i++) {
                ctx.fillStyle = getRandomColor();
                ctx.beginPath();
                ctx.arc(Math.random() * canvas.width, Math.random() * canvas.height, Math.random() * 3, 0, Math.PI * 2);
                ctx.fill();
            }

            ctx.font = '22px estdad';
            ctx.fillStyle = '#333';
            ctx.setTransform(1, Math.random() * 0.3, Math.random() * 0.3, 1, Math.random() * 10, Math.random() * 10);
            ctx.fillText(captcha, 70, 15);
            ctx.setTransform(1, 0, 0, 1, 0, 0);
        }

        function getRandomColor() {
            const letters = '0123456789ABCDEF';
            let color = '#';
            for (let i = 0; i < 6; i++) {
                color += letters[Math.floor(Math.random() * 16)];
            }
            return color;
        }

        let captchaAnswer = generateCaptcha();

        $('#sab-refresh-captcha').click(function (e) {
            e.preventDefault();
            captchaAnswer = generateCaptcha();
            $('#sab-captcha-input').val('');
        });

        $('#sab-appointment-form').submit(function (e) {
            const userAnswer = $('#sab-captcha-input').val();
            if (userAnswer !== captchaAnswer) {
                e.preventDefault();
                Swal.fire({
                    title: 'خطا',
                    text: 'کد امنیتی اشتباه است. لطفاً دوباره تلاش کنید.',
                    icon: 'error',
                    confirmButtonText: 'بازگشت',
                    customClass: {
                        popup: 'swal-rtl',
                    },
                });
                captchaAnswer = generateCaptcha();
            }
        });
    }
});