<?php

namespace Noorgfx;

class Dashboard {
	public function __construct() {
		add_filter(
			'noorgfx:dashboard:redirect-after-activation',
			function ($url) {
				return add_query_arg(
					'page',
					'ct-dashboard',
					admin_url('admin.php')
				);
			}
		);

		add_filter(
			'noorgfx_add_menu_page',
			function ($res, $options) {
				add_menu_page(
					$options['title'],
					$options['menu-title'],
					$options['permision'],
					$options['top-level-handle'],
					$options['callback'],
					$options['icon-url'],
					2
				);

				add_submenu_page(
					$options['top-level-handle'],
					$options['title'],
					__('Dashboard', 'noorgfx-component'),
					$options['permision'],
					$options['top-level-handle']
				);

				return true;
			},
			10, 2
		);

		add_action(
			'admin_menu',
			[$this, 'setup_framework_page'],
			5
		);

		if (defined('WP_FS__LOWEST_PRIORITY')) {
			add_action(
				'network_admin_menu',
				function () {
					global $menu;

					foreach ($menu as $key => $item) {
						if ($item[2] === 'ct-dashboard') {
							$menu[$key][0] = __('Noorgfx', 'noorgfx-component');
							$menu[$key][3] = __('Noorgfx', 'noorgfx-component');
							$menu[$key][4] = "menu-top";
							$menu[$key][6] = set_url_scheme($this->get_icon());
						}
					}
				},
				WP_FS__LOWEST_PRIORITY + 1
			);
		}

		add_filter(
			'noorgfx:dashboard:redirect-after-activation',
			function ($url) {
				return add_query_arg(
					'page',
					'ct-dashboard',
					admin_url('admin.php')
				);
			}
		);

		add_action('admin_notices', function () {
			$noorgfx_data = Plugin::instance()->is_noorgfx_data;

			if (
				! Plugin::instance()->check_if_noorgfx_is_activated()
				&&
				$noorgfx_data
				&&
				$noorgfx_data['is_correct_theme']
			) {
				echo noorgfx_render_view(
					dirname(__FILE__) . '/views/theme-mismatch.php',
					[
						'is_theme_version_ok' => $noorgfx_data['is_theme_version_ok'],
						'is_component_version_ok' => $noorgfx_data['is_component_version_ok'],
					]
				);
			}
		});

		add_action(
			'admin_menu',
			function () {
				if (Plugin::instance()->check_if_noorgfx_is_activated()) {
					return;
				}

				$menu_slug = plugin_basename('ct-dashboard');
				$hookname = get_plugin_page_hookname('ct-dashboard', '');
				remove_all_actions($hookname);

				add_action(
					$hookname,
					function () {
						$this->welcome_page_template();
					}
				);
			},
			99999999999
		);

		add_action(
			'admin_enqueue_scripts',
			[$this, 'enqueue_static'],
			100
		);

		add_action('admin_body_class', function ($class) {
			if (! Plugin::instance()->check_if_noorgfx_is_activated()) {
				return $class;
			}

			if (function_exists('blc_fs') && blc_fs()->is_activation_mode()) {
				$class .= ' noorgfx-fs-optin-dashboard';
			}

			return $class;
		});

		if (function_exists('blc_fs')) {
			blc_fs()->add_filter('hide_plan_change', '__return_true');
			blc_fs()->add_filter(
				'plugin_icon',
				function ($url) {
					return NOORGFX_PATH . '/static/img/logo.jpg';
				}
			);

			blc_fs()->add_filter(
				'permission_diagnostic_default',
				'__return_false'
			);

			blc_fs()->add_filter(
				'show_deactivation_feedback_form',
				'__return_false'
			);

			blc_fs()->add_filter( 'show_deactivation_subscription_cancellation', '__return_false' );

			blc_fs()->add_filter(
				'connect-message_on-premium',
				function ($text) {
					if (strpos($text, '<br>') !== false) {
						$exploded_message = explode('<br>', $text);

						$text = '<span>' . $exploded_message[0] . '</span>' . $exploded_message[1];
					}

					return $text;
				}
			);

			blc_fs()->add_filter(
				'connect_message_on_update',
				function (
					$message,
					$user_first_name,
					$product_title,
					$user_login,
					$site_link,
				) {
					$is_network_upgrade_mode = ( fs_is_network_admin() && blc_fs()->is_network_upgrade_mode() );
					$slug = blc_fs()->get_slug();
					$is_gdpr_required = \FS_GDPR_Manager::instance()->is_required();
					$hey_x_text = esc_html( blc_safe_sprintf( fs_text_x_inline( 'Hey %s,', 'greeting', 'hey-x', $slug ), $user_first_name ) );

					$default_optin_message = $is_gdpr_required ?
						fs_text_inline( 'Never miss an important update - opt in to our security & feature updates notifications, educational content, offers, and non-sensitive diagnostic tracking with %4$s. If you skip this, that\'s okay! %1$s will still work just fine.', 'connect-message_on-update', $slug ) :
						fs_text_inline( 'Never miss an important update - opt in to our security & feature updates notifications, and non-sensitive diagnostic tracking with %4$s. If you skip this, that\'s okay! %1$s will still work just fine.', 'connect-message_on-update', $slug );

					$default_optin_message = 'Never miss an important update - opt in to our security & feature updates notifications, educational content, offers, and non-sensitive diagnostic tracking with.';

					return (($is_network_upgrade_mode ?
						'' :
						/* translators: %s: name (e.g. Hey John,) */
						'<span>' . $hey_x_text . '</span>'
					) .
					blc_safe_sprintf(
						esc_html( $default_optin_message ),
						'<b>' . esc_html( blc_fs()->get_plugin_name() ) . '</b>',
						'<b>' . $user_login . '</b>',
						$site_link,
					));

				}, 10, 6
			);

			blc_fs()->add_action('connect/before', function () {
				$path = dirname(__FILE__) . '/views/optin.php';

				echo noorgfx_render_view(
					$path,
					[]
				);
			});

			blc_fs()->add_action('connect/after', function () {
				echo '</div>';
			});

			add_action(
				'wp_ajax_noorgfx_fs_connect_again',
				function () {
					if (! current_user_can('edit_theme_options')) {
						wp_send_json_error();
					}

					blc_fs()->connect_again();
					wp_send_json_success();
				}
			);

			add_action(
				'wp_ajax_noorgfx_dashboard_handle_incorrect_license',
				function () {
					if (! current_user_can(
						blc_get_capabilities()->get_wp_capability_by('dashboard')
					)) {
						wp_send_json_error();
					}

					$noorgfx_active_extensions_old = get_option(
						'noorgfx_active_extensions_old',
						'__empty__'
					);

					if ($noorgfx_active_extensions_old !== '__empty__') {
						return;
					}

					$activated_extensions = get_option('noorgfx_active_extensions', []);

					update_option(
						'noorgfx_active_extensions_old',
						$activated_extensions
					);

					delete_option('noorgfx_active_extensions');

					wp_send_json_success();
				}
			);
		}

		add_filter(
			'noorgfx_dashboard_localizations',
			function ($d) {
				$plugin_data = get_plugin_data(NOORGFX__FILE__);

				$result = [
					'is_pro' => false,
					'is_anonymous' => false,
					'connect_template' => '',
					'retrieve_demos_data' => [],
					'plugin_version' => $plugin_data['Version'],
					'is_multisite' => is_multisite()
				];

				if (function_exists('blc_fs')) {
					$is_anonymous = blc_fs()->is_anonymous();
					$connect_template = '';

					if ($is_anonymous) {
						ob_start();
						blc_fs()->_connect_page_render();
						$connect_template = ob_get_clean();
					}

					$current_plan = blc_get_capabilities()->get_plan();

					// $current_plan = 'free';

					$retrieve_demos_data = [
					];

					if (blc_fs()->_get_license()) {
						$retrieve_demos_data['license_id'] = blc_fs()->_get_license()->id;
					}

					if (blc_fs()->get_site()) {
						$retrieve_demos_data['install_id'] = blc_fs()->get_site()->id;
					}

					$result = [
						'is_pro' => $current_plan !== 'free',
						'current_plan' => $current_plan,

						'pro_starter_sites' => blc_get_capabilities()->get_features()['pro_starter_sites'],
						'pro_starter_sites_enhanced' => blc_get_capabilities()->get_features()['pro_starter_sites_enhanced'],

						'is_anonymous' => $is_anonymous,
						'connect_template' => $connect_template,
						'retrieve_demos_data' => $retrieve_demos_data,
						'plugin_version' => $plugin_data['Version'],
						'is_multisite' => is_multisite()
					];
				}

				if (
					Plugin::instance()->premium
					&&
					is_callable([
						Plugin::instance()->premium,
						'user_wants_beta_updates'
					])
				) {
					$result['has_beta_consent'] = Plugin::instance()->premium->user_wants_beta_updates();
				}

				if (function_exists('noorgfx_get_pricing_links')) {
					$result['modal_links'] = noorgfx_get_pricing_links();
				}

				return array_merge($result, $d);
			}
		);

		add_action('admin_init', function ($plugin) {
			if (wp_doing_ajax()) {
				return;
			}

			if (! is_admin()) {
				return;
			}

			if (! is_user_logged_in()) {
				return;
			}

			if (is_network_admin()) {
				return;
			}

			if (intval(get_option('blc_activation_redirect', false)) === wp_get_current_user()->ID) {
				delete_option('blc_activation_redirect');
				exit(wp_redirect(admin_url('admin.php?page=ct-dashboard')));
			}
		});
	}

	public function enqueue_static() {
		if (! $this->is_dashboard_page()) {
			$this->enqueue_static_global();
			return;
		}

		$deps = apply_filters('noorgfx-dashboard-scripts-dependencies', [
			'wp-i18n',
			'ct-events',
			'ct-options-scripts'
		]);

		if (Plugin::instance()->check_if_noorgfx_is_activated()) {
			wp_enqueue_script(
				'noorgfx-dashboard-scripts',
				NOORGFX_URL . 'static/bundle/dashboard.js',
				$deps,
				blc_get_version(),
				false
			);
		} else {
			wp_enqueue_script(
				'noorgfx-dashboard-scripts',
				NOORGFX_URL . 'static/bundle/dashboard-no-theme.js',
				[
					'underscore',
					'react',
					'react-dom',
					'wp-element',
					'wp-date',
					'wp-i18n',
					'wp-util'
				],
				blc_get_version(),
				false
			);

			$slug = 'noorgfx';

			$theme_activate_url = null;
			$theme_activate_url_multi_site = null;

			if (current_user_can('switch_themes')) {
				if (is_multisite()) {
					$theme_activate_url_multi_site = add_query_arg(
						[
							'action' => 'enable',
							'_wpnonce' => wp_create_nonce('enable-theme_' . $slug),
							'theme' => $slug
						],
						network_admin_url('themes.php')
					);
				}

				$theme_activate_url = add_query_arg(
					[
						'action' => 'activate',
						'_wpnonce' => wp_create_nonce('switch-theme_' . $slug),
						'stylesheet' => $slug,
					],
					admin_url('themes.php')
				);
			}

			$localize_data = [
				'themeIsInstalled' => (
					!! wp_get_theme($slug)
					&&
					! wp_get_theme($slug)->errors()
				),
				'updatesNonce' => wp_installing() ? '' : wp_create_nonce('updates'),
				'activate_multi_site' => $theme_activate_url_multi_site,
				'activate'=> $theme_activate_url
			];

			$noorgfx_data = Plugin::instance()->is_noorgfx_data;

			if ($noorgfx_data && $noorgfx_data['is_correct_theme']) {
				$mismatched_product_name = 'Noorgfx theme';
				$mismatched_product_slug = 'noorgfx';

				if (
					$noorgfx_data['is_theme_version_ok']
					&&
					! $noorgfx_data['is_component_version_ok']
				) {
					$mismatched_product_name = 'Noorgfx Component plugin';
					$mismatched_product_slug = 'noorgfx-component';

					if (blc_can_use_premium_code()) {
						$mismatched_product_name = 'Noorgfx Component Pro plugin';
						$mismatched_product_slug = 'noorgfx-component-plg';
					}
				}

				$localize_data['theme_version_mismatch'] = [
					'productName' => $mismatched_product_name,
					'slug' => $mismatched_product_slug
				];
			}

			wp_localize_script(
				'noorgfx-dashboard-scripts',
				'ctDashboardLocalizations',
				$localize_data
			);

			wp_dequeue_style('ct-dashboard-styles');
		}

		wp_enqueue_style(
			'noorgfx-dashboard-styles',
			NOORGFX_URL . 'static/bundle/dashboard.min.css',
			['wp-components'],
			blc_get_version()
		);
	}

	public function enqueue_static_global() {
		$slug = 'noorgfx';

		$themeIsInstalled = (
			!! wp_get_theme($slug)
			&&
			! wp_get_theme($slug)->errors()
		);

		$noorgfx_data = Plugin::instance()->is_noorgfx_data;

		if (
			! Plugin::instance()->check_if_noorgfx_is_activated()
			&&
			$noorgfx_data
			&&
			$noorgfx_data['is_correct_theme']
		) {
			wp_enqueue_style(
				'noorgfx-dashboard-styles',
				NOORGFX_URL . 'static/bundle/dashboard.min.css',
				[],
				blc_get_version()
			);

			wp_enqueue_script(
				'noorgfx-admin-notifications-scripts',
				NOORGFX_URL . 'static/bundle/notifications.js',
				[
					'underscore',
					'react',
					'react-dom',
					'wp-element',
					'wp-date',
					'wp-i18n'
				],
				blc_get_version(),
				false
			);

			wp_localize_script(
				'noorgfx-admin-notifications-scripts',
				'ctDashboardLocalizations',
				[
					'updatesNonce' => wp_installing() ? '' : wp_create_nonce('updates'),
				]
			);
		}
	}

	public function get_icon() {
		return apply_filters(
			'noorgfx:dashboard:icon-url',
			'data:image/svg+xml;base64,PD94bWwgdmVyc2lvbj0iMS4wIiBlbmNvZGluZz0idXRmLTgiPz48IS0tIFVwbG9hZGVkIHRvOiBTVkcgUmVwbywgd3d3LnN2Z3JlcG8uY29tLCBHZW5lcmF0b3I6IFNWRyBSZXBvIE1peGVyIFRvb2xzIC0tPg0KPHN2ZyB3aWR0aD0iODAwcHgiIGhlaWdodD0iODAwcHgiIHZpZXdCb3g9IjAgMCAyNCAyNCIgZmlsbD0ibm9uZSIgeG1sbnM9Imh0dHA6Ly93d3cudzMub3JnLzIwMDAvc3ZnIj4NCjxwYXRoIGQ9Ik0xMC45NyAySDguOTdDMy45NyAyIDEuOTcgNCAxLjk3IDlWMTVDMS45NyAyMCAzLjk3IDIyIDguOTcgMjJIMTQuOTdDMTkuOTcgMjIgMjEuOTcgMjAgMjEuOTcgMTVWMTMiIHN0cm9rZT0iIzI5MkQzMiIgc3Ryb2tlLXdpZHRoPSIxLjUiIHN0cm9rZS1saW5lY2FwPSJyb3VuZCIgc3Ryb2tlLWxpbmVqb2luPSJyb3VuZCIvPg0KPHBhdGggZD0iTTIxLjg4IDMuNTU5OThDMjAuNjUgNi42Mjk5OCAxNy41NiAxMC44MSAxNC45OCAxMi44OEwxMy40IDE0LjE0QzEzLjIgMTQuMjkgMTMgMTQuNDEgMTIuNzcgMTQuNUMxMi43NyAxNC4zNSAxMi43NiAxNC4yIDEyLjc0IDE0LjA0QzEyLjY1IDEzLjM3IDEyLjM1IDEyLjc0IDExLjgxIDEyLjIxQzExLjI2IDExLjY2IDEwLjYgMTEuMzUgOS45MiAxMS4yNkM5Ljc2IDExLjI1IDkuNiAxMS4yNCA5LjQ0IDExLjI1QzkuNTMgMTEgOS42NiAxMC43NyA5LjgzIDEwLjU4TDExLjA5IDguOTk5OThDMTMuMTYgNi40MTk5OCAxNy4zNSAzLjMwOTk4IDIwLjQxIDIuMDc5OThDMjAuODggMS44OTk5OCAyMS4zNCAyLjAzOTk4IDIxLjYzIDIuMzI5OThDMjEuOTMgMi42Mjk5OCAyMi4wNyAzLjA4OTk4IDIxLjg4IDMuNTU5OThaIiBzdHJva2U9IiMyOTJEMzIiIHN0cm9rZS13aWR0aD0iMS41IiBzdHJva2UtbGluZWNhcD0icm91bmQiIHN0cm9rZS1saW5lam9pbj0icm91bmQiLz4NCjxwYXRoIGQ9Ik0xMi43OCAxNC40OUMxMi43OCAxNS4zNyAxMi40NCAxNi4yMSAxMS44MSAxNi44NUMxMS4zMiAxNy4zNCAxMC42NiAxNy42OCA5Ljg3IDE3Ljc4TDcuOSAxNy45OUM2LjgzIDE4LjExIDUuOTEgMTcuMiA2LjAzIDE2LjExTDYuMjQgMTQuMTRDNi40MyAxMi4zOSA3Ljg5IDExLjI3IDkuNDUgMTEuMjRDOS42MSAxMS4yMyA5Ljc3IDExLjI0IDkuOTMgMTEuMjVDMTAuNjEgMTEuMzQgMTEuMjcgMTEuNjUgMTEuODIgMTIuMkMxMi4zNiAxMi43NCAxMi42NiAxMy4zNiAxMi43NSAxNC4wM0MxMi43NyAxNC4xOSAxMi43OCAxNC4zNSAxMi43OCAxNC40OVoiIHN0cm9rZT0iIzI5MkQzMiIgc3Ryb2tlLXdpZHRoPSIxLjUiIHN0cm9rZS1saW5lY2FwPSJyb3VuZCIgc3Ryb2tlLWxpbmVqb2luPSJyb3VuZCIvPg0KPHBhdGggZD0iTTE1LjgyIDExLjk3OTlDMTUuODIgOS44ODk5NCAxNC4xMyA4LjE4OTk0IDEyLjAzIDguMTg5OTQiIHN0cm9rZT0iIzI5MkQzMiIgc3Ryb2tlLXdpZHRoPSIxLjUiIHN0cm9rZS1saW5lY2FwPSJyb3VuZCIgc3Ryb2tlLWxpbmVqb2luPSJyb3VuZCIvPg0KPC9zdmc+'
		);
	}

	public function setup_framework_page() {
		if (! current_user_can(blc_get_capabilities()->get_wp_capability_by('dashboard'))) {
			return;
		}

		$options = [
			'title' => __('دنتیران', 'noorgfx-component'),
			'menu-title' => __('دنتیران', 'noorgfx-component'),
			'permision' => blc_get_capabilities()->get_wp_capability_by('dashboard'),
			'top-level-handle' => 'ct-dashboard',
			'callback' => [$this, 'welcome_page_template'],
			'icon-url' => $this->get_icon(),
			'position' => 2,
		];

		add_menu_page(
			$options['title'],
			$options['menu-title'],
			$options['permision'],
			$options['top-level-handle'],
			$options['callback'],
			$options['icon-url'],
			2
		);
	}

	public function is_dashboard_page() {
		global $pagenow;

		if (is_network_admin()) {
			$is_ct_settings =
				// 'themes.php' === $pagenow &&
				isset( $_GET['page'] ) && 'noorgfx-component' === $_GET['page'];

			return $is_ct_settings;
		}

		$is_ct_settings =
			// 'themes.php' === $pagenow &&
			isset( $_GET['page'] ) && 'ct-dashboard' === $_GET['page'];

		return $is_ct_settings;
	}

	public function welcome_page_template() {
		if (! current_user_can(blc_get_capabilities()->get_wp_capability_by('dashboard'))) {
			wp_die(
				esc_html(
					__( 'You do not have sufficient permissions to access this page.', 'noorgfx-component' )
				)
			);
		}

		echo '<div id="ct-dashboard"></div>';
	}
}

if (! function_exists('noorgfx_render_view')) {
	function noorgfx_render_view(
		$file_path,
		$view_variables = [],
		$default_value = ''
	) {
		if (! is_file($file_path)) {
			return $default_value;
		}

		// phpcs:ignore WordPress.PHP.DontExtract.extract_extract
		extract($view_variables, EXTR_REFS);
		unset($view_variables);

		ob_start();
		require $file_path;

		return ob_get_clean();
	}
}
