import { createElement } from '@wordpress/element'

import { __ } from 'ct-i18n'
import { registerBlockType } from '@wordpress/blocks'

import { InspectorControls, useBlockProps } from '@wordpress/block-editor'
import {
	Panel,
	PanelBody,
	PanelRow,
	ToggleControl,
	__experimentalToggleGroupControl as ToggleGroupControl,
	__experimentalToggleGroupControlOption as ToggleGroupControlOption,
} from '@wordpress/components'

const ResetIcon = () => (
	<svg width="11" height="11" viewBox="0 0 15 15" fill="currentColor">
		<path d="M8.5,7.5l4.5,4.5l-1,1L7.5,8.5L3,13l-1-1l4.5-4.5L2,3l1-1l4.5,4.5L12,2l1,1L8.5,7.5z"></path>
	</svg>
)

const ResetIconButton = () => (
	<svg width="12" height="12" viewBox="0 0 15 15" fill="currentColor">
		<path d="M8.5,7.5l4.5,4.5l-1,1L7.5,8.5L3,13l-1-1l4.5-4.5L2,3l1-1l4.5,4.5L12,2l1,1L8.5,7.5z"></path>
	</svg>
)

const Preview = ({ layout, showResetButton }) => {
	const blockProps = useBlockProps({
		className: 'ct-active-filters',
		'data-display-type': layout,
	})

	const resetButton = (
		<a href="#" className="ct-button-ghost">
			<ResetIconButton />
			{__('Reset Filters', 'noorgfx-component')}
		</a>
	)

	if (layout === 'list') {
		return (
			<ul {...blockProps}>
				<li class="ct-active-filter-item">
					<ul>
						<li className="ct-active-filter-name">
							{__('Category', 'noorgfx-component')}
						</li>
						<li>
							<a href="#">
								<ResetIcon />
								{__('Category 1', 'noorgfx-component')}
							</a>
						</li>
						<li>
							<a href="#">
								<ResetIcon />
								{__('Category 2', 'noorgfx-component')}
							</a>
						</li>
					</ul>
				</li>
				<li className="ct-active-filter-item">
					<ul>
						<li className="ct-active-filter-name">
							{__('Attribute', 'noorgfx-component')}
						</li>
						<li>
							<a href="#">
								<ResetIcon />
								{__('Attribute 1', 'noorgfx-component')}
							</a>
						</li>
						<li>
							<a href="#">
								<ResetIcon />
								{__('Attribute 2', 'noorgfx-component')}
							</a>
						</li>
					</ul>
				</li>
				{showResetButton === 'yes' ? (
					<li className="ct-filter-reset">{resetButton}</li>
				) : null}
			</ul>
		)
	}

	return (
		<div {...blockProps}>
			<a href="#">
				<ResetIcon />
				{__('Category', 'noorgfx-component')}
			</a>
			<a href="#">
				<ResetIcon />
				{__('Attribute', 'noorgfx-component')}
			</a>

			{showResetButton === 'yes' ? (
				<div className="ct-filter-reset">{resetButton}</div>
			) : null}
		</div>
	)
}

registerBlockType('noorgfx/active-filters', {
	apiVersion: 3,
	title: __('Active Filters', 'noorgfx-component'),
	description: __(
		'Display the currently active filters.',
		'noorgfx-component'
	),
	icon: 'filter',
	icon: {
		src: (
			<svg
				xmlns="http://www.w3.org/2000/svg"
				viewBox="0 0 24 24"
				className="wc-block-editor-components-block-icon">
				<path d="M18.7,7.1c-0.4-1.5-1.7-2.6-3.3-2.6S12.4,5.6,12,7.1H4v1.8h8c0.4,1.5,1.7,2.5,3.3,2.5s2.9-1.1,3.3-2.5H20V7.1H18.7zM15.3,9.8c-1,0-1.8-0.8-1.8-1.8c0-1,0.8-1.8,1.8-1.8c1,0,1.8,0.8,1.8,1.8C17.1,8.9,16.3,9.8,15.3,9.8z M8.7,12.6c-1.6,0-2.9,1.1-3.3,2.6H4v1.8h1.3c0.4,1.5,1.7,2.5,3.3,2.5s2.9-1.1,3.3-2.5h8v-1.8h-8C11.6,13.7,10.3,12.6,8.7,12.6z M8.7,17.8c-1,0-1.8-0.8-1.8-1.8c0-1,0.8-1.8,1.8-1.8c1,0,1.8,0.8,1.8,1.8C10.5,17,9.7,17.8,8.7,17.8z" />
			</svg>
		),
	},
	category: 'widgets',
	supports: {
		html: false,
		inserter: false,
		lock: false,
	},
	attributes: {
		layout: {
			type: 'string',
			default: 'list',
		},
		showResetButton: {
			type: 'string',
			default: 'yes',
		},
	},
	edit: ({ attributes, setAttributes }) => {
		const { layout, showResetButton } = attributes

		return (
			<>
				<Preview {...attributes} />
				<InspectorControls>
					<Panel header="Filter Settings">
						<PanelBody>
							<ToggleGroupControl
								label={__('Display Type', 'noorgfx-component')}
								value={layout}
								isBlock
								onChange={(newLayout) =>
									setAttributes({ layout: newLayout })
								}>
								<ToggleGroupControlOption
									key="list"
									value="list"
									label={__('List', 'noorgfx-component')}
								/>
								<ToggleGroupControlOption
									key="inline"
									value="inline"
									label={__('Inline', 'noorgfx-component')}
								/>
							</ToggleGroupControl>
							<PanelRow>
								<ToggleControl
									label={__(
										'Show Reset Button',
										'noorgfx-component'
									)}
									help={__(
										'Show or hide reset filter button.',
										'noorgfx-component'
									)}
									checked={showResetButton === 'yes'}
									onChange={() =>
										setAttributes({
											showResetButton:
												showResetButton === 'yes'
													? 'no'
													: 'yes',
										})
									}
								/>
							</PanelRow>
						</PanelBody>
					</Panel>
				</InspectorControls>
			</>
		)
	},
	save: function () {
		return (
			<div class="wp-block-noorgfx-active-filters">
				Noorgfx: Active Filter
			</div>
		)
	},
})

wp.blocks.registerBlockVariation('noorgfx/widgets-wrapper', {
	name: 'noorgfx-active-filters',
	title: __('Active Filters', 'noorgfx-component'),
	attributes: {
		heading: __('Active Filters', 'noorgfx-component'),
		block: 'noorgfx/active-filters',
	},
	isActive: (attributes) => attributes.block === 'noorgfx/active-filters',
	icon: {
		src: (
			<svg
				xmlns="http://www.w3.org/2000/svg"
				viewBox="0 0 24 24"
				className="wc-block-editor-components-block-icon">
				<path d="M18.7,7.1c-0.4-1.5-1.7-2.6-3.3-2.6S12.4,5.6,12,7.1H4v1.8h8c0.4,1.5,1.7,2.5,3.3,2.5s2.9-1.1,3.3-2.5H20V7.1H18.7zM15.3,9.8c-1,0-1.8-0.8-1.8-1.8c0-1,0.8-1.8,1.8-1.8c1,0,1.8,0.8,1.8,1.8C17.1,8.9,16.3,9.8,15.3,9.8z M8.7,12.6c-1.6,0-2.9,1.1-3.3,2.6H4v1.8h1.3c0.4,1.5,1.7,2.5,3.3,2.5s2.9-1.1,3.3-2.5h8v-1.8h-8C11.6,13.7,10.3,12.6,8.7,12.6z M8.7,17.8c-1,0-1.8-0.8-1.8-1.8c0-1,0.8-1.8,1.8-1.8c1,0,1.8,0.8,1.8,1.8C10.5,17,9.7,17.8,8.7,17.8z" />
			</svg>
		),
	},
})
