<?php

namespace Noorgfx\Extensions\WoocommerceExtra;

class WooTermsImportExport {
	public function __construct() {
		add_filter(
			'woocommerce_product_export_row_data',
			[$this, 'export_terms_meta'],
			10,
			3
		);

		add_filter(
			'woocommerce_product_import_inserted_product_object',
			[$this, 'set_taxonomy_meta'],
			10,
			2
		);

		add_filter(
			'noorgfx_woo_product_export:finalize',
			[$this, 'merge_unique_product_data'],
		);
	}

	public function export_terms_meta($row, $product, $exporter) {
		$categories = get_the_terms($product->get_id(), 'product_cat');
		$brands = get_the_terms($product->get_id(), 'product_brand');

		$formatted_categories = [];
		$formatted_brands = [];
		$formatted_attributes = [];

		if (
			! is_wp_error($categories)
			&&
			is_array($categories)
		) {
			foreach ($categories as $category) {
				$category_meta = get_term_meta(
					$category->term_id,
					'noorgfx_taxonomy_meta_options',
					true
				);

				$thumbnail_id = get_term_meta($category->term_id, 'thumbnail_id', true);

				$formatted_categories[] = [
					'name' => $category->name,
					'thumb' => wp_get_attachment_url($thumbnail_id),
					'meta' => $category_meta
				];
			}
		}

		$attributes = $product->get_attributes();
		$attribute_taxonomies = wc_get_attribute_taxonomies();

		if (
			! is_wp_error($attribute_taxonomies)
			&&
			is_array($attribute_taxonomies)
		) {
			foreach ($attribute_taxonomies as $attribute) {
				if (
					! in_array(
						wc_attribute_taxonomy_name(
							$attribute->attribute_name
						),
						array_keys($attributes)
					)
				) {
					continue;
				}

				$attribute_for_product = $attributes[wc_attribute_taxonomy_name($attribute->attribute_name)];

				if (! isset($attribute_for_product['options'])) {
					continue;
				}

				$terms = get_terms([
					'taxonomy' => wc_attribute_taxonomy_name($attribute->attribute_name),
					'hide_empty' => false,
					'include' => $attribute_for_product['options']
				]);

				$children_data = [];

				foreach ($terms as $term) {
					$term_meta = get_term_meta(
						$term->term_id,
						'noorgfx_taxonomy_meta_options',
						true
					);

					$children_data[] = [
						'name' => $term->name,
						'meta' => $term_meta
					];
				}

				$formatted_attributes[] = [
					'name' => $attribute->attribute_label,
					'slug' => $attribute->attribute_name,
					'type' => $attribute->attribute_type,
					'children' => $children_data
				];
			}
		}

		if (
			! is_wp_error($brands)
			&&
			is_array($brands)
		) {
			foreach ($brands as $brand) {
				$brand_meta = get_term_meta(
					$brand->term_id,
					'noorgfx_taxonomy_meta_options',
					true
				);

				$thumbnail_id = get_term_meta($brand->term_id, 'thumbnail_id', true);

				if (
					isset($brand_meta['image']['attachment_id'])
					&&
					! empty($brand_meta['image']['attachment_id'])
				) {
					$brand_meta['image'] = [
						'attachment_id' => $brand_meta['image']['attachment_id'],
						'url' => wp_get_attachment_url($brand_meta['image']['attachment_id'])
					];
				}

				$formatted_brands[] = [
					'name' => $brand->name,
					'thumb' => wp_get_attachment_url($thumbnail_id),
					'meta' => $brand_meta
				];
			}
		}

		$custom_data = blc_theme_functions()->noorgfx_manager()
			->woocommerce
			->import_export
			->get_custom_data();

		if (! isset($custom_data['noorgfx_product_categories'])) {
			$custom_data['noorgfx_product_categories'] = [];
		}

		if (! isset($custom_data['noorgfx_product_attributes'])) {
			$custom_data['noorgfx_product_attributes'] = [];
		}

		if (! isset($custom_data['noorgfx_product_brands'])) {
			$custom_data['noorgfx_product_brands'] = [];
		}

		$custom_data['noorgfx_product_categories'] = array_merge(
			$custom_data['noorgfx_product_categories'],
			$formatted_categories
		);

		$custom_data['noorgfx_product_attributes'] = array_merge(
			$custom_data['noorgfx_product_attributes'],
			$formatted_attributes
		);

		$custom_data['noorgfx_product_brands'] = array_merge(
			$custom_data['noorgfx_product_brands'],
			$formatted_brands
		);

		blc_theme_functions()->noorgfx_manager()
			->woocommerce
			->import_export
			->set_custom_data($custom_data);

		return $row;
	}

	public function merge_unique_product_data($data) {
		$result = [
			'noorgfx_product_categories' => [],
			'noorgfx_product_attributes' => [],
			'noorgfx_product_brands' => [],
		];

		if (! empty($data['noorgfx_product_categories'])) {
			$seen = [];

			foreach ($data['noorgfx_product_categories'] as $cat) {
				$key = $cat['name'] . '|' . $cat['thumb'];
				$seen[$key] = $cat;
			}

			$result['noorgfx_product_categories'] = array_values($seen);
		}

		if (! empty($data['noorgfx_product_attributes'])) {
			$merged = [];

			foreach ($data['noorgfx_product_attributes'] as $attr) {
				$slug = $attr['slug'];

				if (! isset($merged[$slug])) {
					$merged[$slug] = $attr;
				} else {
					$children = $merged[$slug]['children'];

					foreach ($attr['children'] as $child) {
						$children[$child['name']] = $child;
					}

					$merged[$slug]['children'] = array_values($children);
				}
			}

			$result['noorgfx_product_attributes'] = array_values($merged);
		}

		if (! empty($data['noorgfx_product_brands'])) {
			$seen = [];

			foreach ($data['noorgfx_product_brands'] as $brand) {
				$key = $brand['name'] . '|' . $brand['thumb'];
				$seen[$key] = $brand;
			}

			$result['noorgfx_product_brands'] = array_values($seen);
		}

		return $result;
	}

	public function set_taxonomy_meta($product, $data) {
		if (! $product instanceof \WC_Product) {
			return $product;
		}

		$parsed_data = \Noorgfx\WooImportExport::get_import_file_data();

		$parsed_categories_data = noorgfx_akg('noorgfx_product_categories', $parsed_data, []);
		$parsed_brands_data = noorgfx_akg('noorgfx_product_brands', $parsed_data, []);
		$parsed_attributes_data = noorgfx_akg('noorgfx_product_attributes', $parsed_data, []);

		$product_id = $product->get_id();

		if (! empty($parsed_categories_data)) {
			$categories = [];

			foreach ($parsed_categories_data as $category_data) {
				$category = get_term_by('name', $category_data['name'], 'product_cat');

				if (! $category) {
					continue;
				}

				$category_id = $category->term_id;

				if (
					isset($category_data['thumb'])
					&&
					! empty($category_data['thumb'])
				) {
					$image = \Noorgfx\WooImportExport::get_attachment_id_from_url($category_data['thumb'], $product->get_id());

					if (! is_wp_error($image)) {
						update_term_meta($category_id, 'thumbnail_id', $image);
					}
				}

				if (
					isset($category_data['meta'])
					&&
					! get_term_meta($category_id, 'noorgfx_taxonomy_meta_options', true)
				) {
					update_term_meta($category_id, 'noorgfx_taxonomy_meta_options', $category_data['meta']);
				}
			}
		}

		if (! empty($parsed_brands_data)) {
			$brands = [];

			foreach ($parsed_brands_data as $brand_data) {
				$brand = get_term_by('name', $brand_data['name'], 'product_brand');

				if (! $brand) {
					$brand_id = wp_insert_term($brand_data['name'], 'product_brand');

					if (is_wp_error($brand_id)) {
						continue;
					}

					$brand = get_term($brand_id);
				}

				$brand_id = $brand->term_id;

				if (
					isset($brand_data['thumb'])
					&&
					! empty($brand_data['thumb'])
				) {
					$image = \Noorgfx\WooImportExport::get_attachment_id_from_url($brand_data['thumb'], $product->get_id());
					if (! is_wp_error($image)) {
						update_term_meta($brand_id, 'thumbnail_id', $image);
					}
				}

				if (
					isset($brand_data['meta'])
					&&
					! get_term_meta($brand_id, 'noorgfx_taxonomy_meta_options', true)
				) {
					if (isset($brand_data['meta']['image'])) {
						$image = \Noorgfx\WooImportExport::get_attachment_id_from_url($brand_data['meta']['image']['url'], $product->get_id());

						if (! is_wp_error($image)) {
							$brand_data['meta']['image'] = [
								'attachment_id' => $image,
								'url' => wp_get_attachment_url($image)
							];
						}
					}

					update_term_meta($brand_id, 'noorgfx_taxonomy_meta_options', $brand_data['meta']);
				}
			}
		}

		if (! empty($parsed_attributes_data)) {
			// Drop attributes cache so we can update them properly.
			wp_schedule_single_event(time(), 'woocommerce_flush_rewrite_rules');
			delete_transient('wc_attribute_taxonomies');
			\WC_Cache_Helper::invalidate_cache_group('woocommerce-attributes');

			foreach (array_values(wc_get_attribute_taxonomies()) as $tax) {
				$taxonomy = (array) $tax;

				foreach ($parsed_attributes_data as $attribute_data) {
					if ($attribute_data['name'] !== $taxonomy['attribute_label']) {
						continue;
					}

					wc_update_attribute(
						$taxonomy['attribute_id'],
						[
							'type' => $attribute_data['type'] ?? 'select',
						]
					);
				}
			}

			foreach ($parsed_attributes_data as $attribute_data) {
				if (
					isset($attribute_data['children'])
					&&
					! empty($attribute_data['children'])
				) {
					foreach ($attribute_data['children'] as $term_data) {
						$term = get_term_by('name', $term_data['name'], wc_attribute_taxonomy_name($attribute_data['slug']));

						if (! $term) {
							continue;
						}

						$term_id = $term->term_id;

						if (
							isset($term_data['meta'])
							&&
							! get_term_meta($term_id, 'noorgfx_taxonomy_meta_options', true)
						) {

							if (isset($term_data['meta']['image'])) {
								$image = \Noorgfx\WooImportExport::get_attachment_id_from_url($term_data['meta']['image']['url'], $product->get_id());

								if (! is_wp_error($image)) {
									$term_data['meta']['image'] = [
										'attachment_id' => $image,
										'url' => wp_get_attachment_url($image)
									];
								}
							}

							update_term_meta($term_id, 'noorgfx_taxonomy_meta_options', $term_data['meta']);
						}
					}
				}
			}
		}

		return $product;
	}
}
