<?php
if ( ! defined( 'ABSPATH' ) ) {
    exit;
}

// Services
function sab_get_services() {
    global $wpdb;
    $table_name = $wpdb->prefix . 'sab_services';
    $services = $wpdb->get_results("SELECT * FROM $table_name");
    if ($services) {
        wp_send_json_success($services);
    } else {
        wp_send_json_error('No services found');
    }
}
add_action('wp_ajax_sab_get_services', 'sab_get_services');
add_action('wp_ajax_nopriv_sab_get_services', 'sab_get_services');

// Providers
function sab_get_doctors() {
    global $wpdb;
    $service_id = isset($_POST['service_id']) ? intval($_POST['service_id']) : 0;
    $table_name = $wpdb->prefix . 'sab_doctors';
    
    if (empty($service_id)) {
        $doctors = $wpdb->get_results("SELECT * FROM $table_name WHERE is_independent = 1");
    } else {
        $doctors = $wpdb->get_results($wpdb->prepare("SELECT * FROM $table_name WHERE service_id = %d OR is_independent = 1", $service_id));
    }

    if ($doctors) {
        wp_send_json_success($doctors);
    } else {
        wp_send_json_error('No doctors found');
    }
}
add_action('wp_ajax_sab_get_doctors', 'sab_get_doctors');
add_action('wp_ajax_nopriv_sab_get_doctors', 'sab_get_doctors');

// Availability
function sab_get_availability() {
    global $wpdb;
    $doctor_id = intval($_POST['doctor_id']);
    $table_name = $wpdb->prefix . 'sab_availability';
    $availability = $wpdb->get_results($wpdb->prepare("SELECT * FROM $table_name WHERE doctor_id = %d", $doctor_id));

    if ($availability) {
        $dates = array();
        $times = array();
        foreach ($availability as $slot) {
            $start_date = new DateTime($slot->date);
            $end_date = new DateTime($slot->end_date ?: $slot->date);

            $interval = new DateInterval('P1D');
            $date_range = new DatePeriod($start_date, $interval, $end_date->modify('+1 day'));

            foreach ($date_range as $date) {
                $formatted_date = $date->format('Y-m-d');
                $dates[] = $formatted_date;
                $times[$formatted_date] = explode(',', $slot->times);
            }
        }
        $dates = array_unique($dates);
        wp_send_json_success(array('dates' => $dates, 'times' => $times));
    } else {
        wp_send_json_error('No availability found');
    }
}
add_action('wp_ajax_sab_get_availability', 'sab_get_availability');
add_action('wp_ajax_nopriv_sab_get_availability', 'sab_get_availability');

// Convert to Jalali
function convert_to_jalali($date) {
    if (class_exists('IntlDateFormatter')) {
        $formatter = new IntlDateFormatter(
            'fa_IR@calendar=persian', 
            IntlDateFormatter::FULL, 
            IntlDateFormatter::NONE, 
            'Asia/Tehran', 
            IntlDateFormatter::TRADITIONAL
        );
        return $formatter->format(strtotime($date));
    } else {
        return $date; 
    }
}

// Send Notifications
function sab_send_emails($appointment, $is_reminder = false, $reminder_type = '') {
    $user_email_subject = $is_reminder ? get_option('sab_user_email_subject_' . $reminder_type, 'یادآوری رزرو نوبت') : get_option('sab_user_email_subject', 'تایید رزرو نوبت');
    $user_email_body = $is_reminder ? get_option('sab_user_email_body_' . $reminder_type, 'یادآوری: با تشکر از شما برای رزرو نوبت.') : get_option('sab_user_email_body', 'با تشکر از شما برای رزرو نوبت.');

    $day_name = jdate('l', strtotime($appointment['day']));
    $day_number = jdate('j', strtotime($appointment['day']));
    $month_name = jdate('F', strtotime($appointment['day']));
    $year = jdate('Y', strtotime($appointment['day']));
    $formatted_date = "{$day_name} {$day_number} {$month_name} {$year}";

    $replacements = array(
        '{name}' => esc_html($appointment['patient_name']),
        '{phone}' => esc_html($appointment['patient_phone']),
        '{mobile}' => esc_html($appointment['patient_mobile']),
        '{email}' => esc_html($appointment['patient_email']),
        '{service}' => esc_html($appointment['service_name']),
        '{doctor}' => esc_html($appointment['doctor_name']),
        '{date}' => esc_html($formatted_date),
        '{time}' => esc_html($appointment['time']),
        '{custom-1}' => esc_html($appointment['custom_text_1']),
        '{custom-2}' => esc_html($appointment['custom_text_2']),
        '{custom-3}' => esc_html($appointment['custom_text_3']),
        '{custom-4}' => esc_html($appointment['custom_text_4']),
    );

    $user_email_body = str_replace(array_keys($replacements), array_values($replacements), $user_email_body);
    $user_email_body = wpautop($user_email_body);

    $headers = array('Content-Type: text/html; charset=UTF-8');

    if (!$is_reminder) {
        $admin_email = get_option('sab_admin_email', get_option('admin_email'));
        $admin_email_subject = get_option('sab_admin_email_subject', 'رزرو نوبت جدید');
        $admin_email_body = get_option('sab_admin_email_body', 'یک رزرو نوبت جدید ثبت شده است.');
        $admin_email_body = str_replace(array_keys($replacements), array_values($replacements), $admin_email_body);
        $admin_email_body = wpautop($admin_email_body);

        $admin_emails = array_map('trim', explode(',', $admin_email));
        $valid_admin_emails = array_filter($admin_emails, 'is_email');

        if (!empty($valid_admin_emails)) {
            wp_mail($valid_admin_emails, $admin_email_subject, $admin_email_body, $headers);
        } else {
            error_log('No valid admin emails provided for sending notification.');
        }
    }

    if (is_email($appointment['patient_email'])) {
        wp_mail($appointment['patient_email'], $user_email_subject, $user_email_body, $headers);
    } else {
        error_log('Invalid user email provided: ' . $appointment['patient_email']);
    }
}
add_filter('wp_mail_content_type', 'sab_set_html_mail_content_type');
function sab_set_html_mail_content_type() {
    return 'text/html';
}
add_action('wp_mail_failed', 'sab_reset_mail_content_type');
function sab_reset_mail_content_type($wp_error) {
    remove_filter('wp_mail_content_type', 'sab_set_html_mail_content_type');
}

// Bookings
function sab_book_appointment() {
    global $wpdb;
    
    $service_id = isset($_POST['service_id']) ? intval($_POST['service_id']) : 0;
    $doctor_id = isset($_POST['doctor_id']) ? intval($_POST['doctor_id']) : 0;
    $day = isset($_POST['date']) ? sanitize_text_field($_POST['date']) : '';
    $time = isset($_POST['time']) ? sanitize_text_field($_POST['time']) : '';
    $name = isset($_POST['name']) ? sanitize_text_field($_POST['name']) : '';
    $phone = isset($_POST['phone']) ? sanitize_text_field($_POST['phone']) : '';
    $mobile = isset($_POST['mobile']) ? sanitize_text_field($_POST['mobile']) : '';
    $email = isset($_POST['email']) ? sanitize_email($_POST['email']) : '';
    $custom_text_1 = isset($_POST['custom_text_1']) ? sanitize_text_field($_POST['custom_text_1']) : '';
    $custom_text_2 = isset($_POST['custom_text_2']) ? sanitize_text_field($_POST['custom_text_2']) : '';
    $custom_text_3 = isset($_POST['custom_text_3']) ? sanitize_text_field($_POST['custom_text_3']) : '';
    $custom_text_4 = isset($_POST['custom_text_4']) ? sanitize_text_field($_POST['custom_text_4']) : '';

    $radio_field = isset($_POST['radio_field']) ? sanitize_text_field($_POST['radio_field']) : '';
    $select_field = isset($_POST['select_field']) ? sanitize_text_field($_POST['select_field']) : '';
    $checkbox_field = '';
    if (isset($_POST['checkbox_field']) && is_array($_POST['checkbox_field'])) {
        $checkbox_field = implode(', ', array_map('sanitize_text_field', $_POST['checkbox_field']));
    }

    if (empty($doctor_id) || empty($day) || empty($time)) {
        wp_send_json_error('داده های ضروری (سرویس دهنده، تاریخ یا زمان) وارد نشده است.');
        return;
    }

    if (empty($service_id)) {
        $is_independent = $wpdb->get_var($wpdb->prepare(
            "SELECT is_independent FROM {$wpdb->prefix}sab_doctors WHERE id = %d",
            $doctor_id
        ));
        if (!$is_independent) {
            wp_send_json_error('سرویس دهنده انتخاب شده مستقل نیست و نیاز به انتخاب خدمات دارد.');
            return;
        }
        $service_id = null; 
    }

    $table_name = $wpdb->prefix . 'sab_appointments';
    $existing = $wpdb->get_var($wpdb->prepare(
        "SELECT COUNT(*) FROM $table_name WHERE doctor_id = %d AND day = %s AND time = %s",
        $doctor_id, $day, $time
    ));
    if ($existing > 0) {
        wp_send_json_error('این زمان قبلاً رزرو شده است.');
        return;
    }
	//uploads
    $file_urls = [];
    for ($i = 0; $i < 10; $i++) {
        if (!empty($_FILES["file_upload_$i"]["name"])) {
            $allowed_file_types = ["image/png", "image/jpeg" , "image/jpg", "application/pdf"];
            $max_file_size = 2 * 1024 * 1024;

            $file_type = $_FILES["file_upload_$i"]["type"];
            $file_size = $_FILES["file_upload_$i"]["size"];

            if (!in_array($file_type, $allowed_file_types)) {
                wp_send_json_error("فرمت فایل مجاز نیست. لطفاً فایل با فرمت PNG, JPEG, JPG یا PDF آپلود کنید.");
                return;
            }

            if ($file_size > $max_file_size) {
                wp_send_json_error("حجم فایل بیش از حد مجاز است. لطفاً فایل کمتر از 2 مگابایت آپلود کنید.");
                return;
            }

            $upload_dir = wp_upload_dir();
            $target_dir = $upload_dir["basedir"] . "/uploads/";
            if (!file_exists($target_dir)) {
                mkdir($target_dir, 0755, true);
            }

            $file_name = basename($_FILES["file_upload_$i"]["name"]);
            $target_file = $target_dir . time() . '-' . $i . '-' . $file_name;

            if (move_uploaded_file($_FILES["file_upload_$i"]["tmp_name"], $target_file)) {
                $file_urls[] = $upload_dir["baseurl"] . "/uploads/" . time() . '-' . $i . '-' . $file_name;
            } else {
                wp_send_json_error("خطا در آپلود فایل شماره " . $i . ".");
                return;
            }
        }
    }

    $result = $wpdb->insert($table_name, array(
        'service_id' => $service_id,
        'doctor_id' => $doctor_id,
        'day' => $day,
        'time' => $time,
        'patient_name' => $name,
        'patient_phone' => $phone,
        'patient_mobile' => $mobile,
        'patient_email' => $email,
        'radio_field' => $radio_field,
        'checkbox_field' => $checkbox_field,
        'select_field' => $select_field,
        'file_path' => json_encode($file_urls),
        'created_at' => current_time('mysql'),
        'custom_text_1' => $custom_text_1,
        'custom_text_2' => $custom_text_2,
        'custom_text_3' => $custom_text_3,
        'custom_text_4' => $custom_text_4,
    ));

    if ($result === false) {
        wp_send_json_error('خطا در ذخیره نوبت در دیتابیس: ' . $wpdb->last_error);
        return;
    }

    $appointment = array(
        'patient_name' => $name,
        'patient_phone' => $phone,
        'patient_mobile' => $mobile,
        'patient_email' => $email,
        'radio_field' => $radio_field,
        'checkbox_field' => $checkbox_field,
        'select_field' => $select_field,
        'custom_text_1' => $custom_text_1,
        'custom_text_2' => $custom_text_2,
        'custom_text_3' => $custom_text_3,
        'custom_text_4' => $custom_text_4,
        'service_name' => $service_id ? $wpdb->get_var($wpdb->prepare("SELECT name FROM {$wpdb->prefix}sab_services WHERE id = %d", $service_id)) : 'نامشخص',
        'doctor_name' => $wpdb->get_var($wpdb->prepare("SELECT name FROM {$wpdb->prefix}sab_doctors WHERE id = %d", $doctor_id)),
        'day' => $day,
        'time' => $time
    );

    sab_send_emails($appointment);
	
	// Schedule reminders
	$appointment_datetime = new DateTime("$day $time", new DateTimeZone('Asia/Tehran'));

	// 24-hour reminder time
	$reminder_24h = clone $appointment_datetime;
	$reminder_24h->modify('-24 hours');
	if ($reminder_24h > new DateTime('now', new DateTimeZone('Asia/Tehran'))) {
		if (!wp_next_scheduled('sab_24h_reminder_event', array($wpdb->insert_id))) {
			wp_schedule_single_event($reminder_24h->getTimestamp(), 'sab_24h_reminder_event', array($wpdb->insert_id));
		}
	}

	// 2-hour reminder time
	$reminder_2h = clone $appointment_datetime;
	$reminder_2h->modify('-2 hours');
	if ($reminder_2h > new DateTime('now', new DateTimeZone('Asia/Tehran'))) {
		if (!wp_next_scheduled('sab_2h_reminder_event', array($wpdb->insert_id))) {
			wp_schedule_single_event($reminder_2h->getTimestamp(), 'sab_2h_reminder_event', array($wpdb->insert_id));
		}
	}

    // Send SMS
    $sms_username = get_option('sab_sms_username');
    $sms_password = get_option('sab_sms_password');
    $sms_from = get_option('sab_sms_from');

    if (!empty($sms_username) && !empty($sms_password) && !empty($sms_from)) {
        $sms_user_message = get_option('sab_sms_user_message');
        $sms_admin_message = get_option('sab_sms_admin_message');
        $sms_doctor_message = get_option('sab_sms_doctor_message');

        $sms_use_template_user = get_option('sab_sms_use_template_user', 0);
        $sms_template_id_user = get_option('sab_sms_template_id_user', '');
        $sms_use_template_admin = get_option('sab_sms_use_template_admin', 0);
        $sms_template_id_admin = get_option('sab_sms_template_id_admin', '');
        $sms_use_template_provider = get_option('sab_sms_use_template_provider', 0);
        $sms_template_id_provider = get_option('sab_sms_template_id_provider', '');

        $admin_mobile = get_option('sab_sms_admin_mobile');
        $doctor_mobile = $wpdb->get_var($wpdb->prepare("SELECT mobile FROM {$wpdb->prefix}sab_doctors WHERE id = %d", $doctor_id));

        $day_name = jdate('l', strtotime($appointment['day']));
        $day_number = jdate('j', strtotime($appointment['day']));
        $month_name = jdate('F', strtotime($appointment['day']));
        $year = jdate('Y', strtotime($appointment['day']));
        $formatted_date = "{$day_name} {$day_number} {$month_name} {$year}";

        $replacements = array(
            '{name}' => $name,
            '{service}' => $appointment['service_name'],
            '{doctor}' => $appointment['doctor_name'],
            '{date}' => $formatted_date,
            '{time}' => $time,
            '{mobile}' => $mobile,
            '{phone}' => $phone,
            '{email}' => $email,
            '{custom-1}' => $custom_text_1,
            '{custom-2}' => $custom_text_2,
            '{custom-3}' => $custom_text_3,
            '{custom-4}' => $custom_text_4,
        );

        $send_sms = function($to, $message, $use_template, $template_id) use ($sms_username, $sms_password, $sms_from) {
            if ($use_template && !empty($template_id)) {
                $data = array(
                    'username' => $sms_username,
                    'password' => $sms_password,
                    'text' => $message,
                    'to' => $to,
                    'bodyId' => $template_id
                );
                $post_data = http_build_query($data);
                $handle = curl_init('https://rest.payamak-panel.com/api/SendSMS/BaseServiceNumber');
                curl_setopt($handle, CURLOPT_HTTPHEADER, array('content-type' => 'application/x-www-form-urlencoded'));
                curl_setopt($handle, CURLOPT_RETURNTRANSFER, true);
                curl_setopt($handle, CURLOPT_SSL_VERIFYHOST, false);
                curl_setopt($handle, CURLOPT_SSL_VERIFYPEER, false);
                curl_setopt($handle, CURLOPT_POST, true);
                curl_setopt($handle, CURLOPT_POSTFIELDS, $post_data);
                $response = curl_exec($handle);
                error_log('Template SMS Response: ' . print_r($response, true));
            } else {
                $data = array(
                    'username' => $sms_username,
                    'password' => $sms_password,
                    'to' => $to,
                    'from' => $sms_from,
                    'text' => $message
                );
                $post_data = http_build_query($data);
                $handle = curl_init('https://rest.payamak-panel.com/api/SendSMS/SendSMS');
                curl_setopt($handle, CURLOPT_HTTPHEADER, array('content-type' => 'application/x-www-form-urlencoded'));
                curl_setopt($handle, CURLOPT_RETURNTRANSFER, true);
                curl_setopt($handle, CURLOPT_SSL_VERIFYHOST, false);
                curl_setopt($handle, CURLOPT_SSL_VERIFYPEER, false);
                curl_setopt($handle, CURLOPT_POST, true);
                curl_setopt($handle, CURLOPT_POSTFIELDS, $post_data);
                $response = curl_exec($handle);
                curl_close($handle);
                error_log('Regular SMS Sent Response: ' . print_r($response, true));
            }
        };

        $user_message = str_replace(array_keys($replacements), array_values($replacements), $sms_user_message);
        $send_sms($mobile, $user_message, $sms_use_template_user, $sms_template_id_user);

        if (!empty($admin_mobile)) {
            $admin_message = str_replace(array_keys($replacements), array_values($replacements), $sms_admin_message);
            $send_sms($admin_mobile, $admin_message, $sms_use_template_admin, $sms_template_id_admin);
        }

        if (!empty($doctor_mobile)) {
            $doctor_message = str_replace(array_keys($replacements), array_values($replacements), $sms_doctor_message);
            $send_sms($doctor_mobile, $doctor_message, $sms_use_template_provider, $sms_template_id_provider);
        }
    }

    // FarazSMS
    $ippanel_username = get_option('sab_ippanel_username');
    $ippanel_password = get_option('sab_ippanel_password');
    $ippanel_from = get_option('sab_ippanel_from');

    $ippanel_user_message = get_option('sab_ippanel_user_message');
    $ippanel_admin_message = get_option('sab_ippanel_admin_message');
    $ippanel_doctor_message = get_option('sab_ippanel_doctor_message');

    if (!empty($ippanel_username) && !empty($ippanel_password) && !empty($ippanel_from)) {
        $day_name = jdate('l', strtotime($appointment['day']));
        $day_number = jdate('j', strtotime($appointment['day']));
        $month_name = jdate('F', strtotime($appointment['day']));
        $year = jdate('Y', strtotime($appointment['day']));
        $formatted_date = "{$day_name} {$day_number} {$month_name} {$year}";

        $replacements = array(
            '{name}' => $name,
            '{service}' => $appointment['service_name'],
            '{doctor}' => $appointment['doctor_name'],
            '{date}' => $formatted_date,
            '{time}' => $time,
            '{mobile}' => $mobile,
            '{phone}' => $phone,
            '{email}' => $email,
            '{custom-1}' => $custom_text_1,
            '{custom-2}' => $custom_text_2,
            '{custom-3}' => $custom_text_3,
            '{custom-4}' => $custom_text_4,
        );

        // User
        $user_message = str_replace(array_keys($replacements), array_values($replacements), $ippanel_user_message);

        // Admin
        $admin_message = str_replace(array_keys($replacements), array_values($replacements), $ippanel_admin_message);

        // Provider
        $doctor_message = str_replace(array_keys($replacements), array_values($replacements), $ippanel_doctor_message);

        $url = "https://ippanel.com/services.jspd";

        // User
        $rcpt_nm_user = array($mobile);
        $param_user = array(
            'uname' => $ippanel_username,
            'pass' => $ippanel_password,
            'from' => $ippanel_from,
            'message' => $user_message,
            'to' => json_encode($rcpt_nm_user),
            'op' => 'send'
        );
        $handler_user = curl_init($url);
        curl_setopt($handler_user, CURLOPT_CUSTOMREQUEST, "POST");
        curl_setopt($handler_user, CURLOPT_POSTFIELDS, $param_user);
        curl_setopt($handler_user, CURLOPT_RETURNTRANSFER, true);
        curl_exec($handler_user);

        // Admin
        $ippanel_admin_mobile = get_option('sab_ippanel_admin_mobile');
        if (!empty($ippanel_admin_mobile)) {
            $rcpt_nm_admin = array($ippanel_admin_mobile);
            $param_admin = array(
                'uname' => $ippanel_username,
                'pass' => $ippanel_password,
                'from' => $ippanel_from,
                'message' => $admin_message,
                'to' => json_encode($rcpt_nm_admin),
                'op' => 'send'
            );
            $handler_admin = curl_init($url);
            curl_setopt($handler_admin, CURLOPT_CUSTOMREQUEST, "POST");
            curl_setopt($handler_admin, CURLOPT_POSTFIELDS, $param_admin);
            curl_setopt($handler_admin, CURLOPT_RETURNTRANSFER, true);
            curl_exec($handler_admin);
        }

        // Provider
        $doctor_faraz_mobile = $wpdb->get_var($wpdb->prepare("SELECT mobile FROM {$wpdb->prefix}sab_doctors WHERE id = %d", $doctor_id));
        if (empty($doctor_faraz_mobile)) {
            $doctor_faraz_mobile = get_option('sab_doctor_faraz_mobile');
        }
        if (!empty($doctor_faraz_mobile)) {
            $rcpt_nm_doctor = array($doctor_faraz_mobile);
            $param_doctor = array(
                'uname' => $ippanel_username,
                'pass' => $ippanel_password,
                'from' => $ippanel_from,
                'message' => $doctor_message,
                'to' => json_encode($rcpt_nm_doctor),
                'op' => 'send'
            );
            $handler_doctor = curl_init($url);
            curl_setopt($handler_doctor, CURLOPT_CUSTOMREQUEST, "POST");
            curl_setopt($handler_doctor, CURLOPT_POSTFIELDS, $param_doctor);
            curl_setopt($handler_doctor, CURLOPT_RETURNTRANSFER, true);
            curl_exec($handler_doctor);
        }
    }

    wp_send_json_success('نوبت شما با موفقیت رزرو شد');
}
add_action('wp_ajax_sab_book_appointment', 'sab_book_appointment');
add_action('wp_ajax_nopriv_sab_book_appointment', 'sab_book_appointment');

// Booked Appointments
function sab_get_booked_appointments() {
    if (isset($_POST['doctor_id'])) {
        $doctor_id = intval($_POST['doctor_id']);

        global $wpdb;
        $appointments_table = $wpdb->prefix . 'sab_appointments'; 
        $results = $wpdb->get_results($wpdb->prepare(
            "SELECT day, time FROM $appointments_table WHERE doctor_id = %d",
            $doctor_id
        ));
        if (!empty($results)) {
            $booked_appointments = array();
            foreach ($results as $appointment) {
                $booked_appointments[] = array(
                    'day' => $appointment->day,
                    'time' => $appointment->time
                );
            }
            wp_send_json_success($booked_appointments);
        } else {
            wp_send_json_success(array());
        }
    } else {
        wp_send_json_error('doctor_id is missing');
    }
}
add_action('wp_ajax_sab_get_booked_appointments', 'sab_get_booked_appointments');
add_action('wp_ajax_nopriv_sab_get_booked_appointments', 'sab_get_booked_appointments');

// Reminder Functions
function sab_send_reminders($appointment_id, $reminder_type) {
    global $wpdb;
    $table_name = $wpdb->prefix . 'sab_appointments';
    $appointment = $wpdb->get_row($wpdb->prepare(
        "SELECT * FROM $table_name WHERE id = %d",
        $appointment_id
    ));

    if (!$appointment) {
        return;
    }

    $service_name = $appointment->service_id ? $wpdb->get_var($wpdb->prepare("SELECT name FROM {$wpdb->prefix}sab_services WHERE id = %d", $appointment->service_id)) : 'نامشخص';
    $doctor_name = $wpdb->get_var($wpdb->prepare("SELECT name FROM {$wpdb->prefix}sab_doctors WHERE id = %d", $appointment->doctor_id));

    $appointment_data = array(
        'patient_name' => $appointment->patient_name,
        'patient_phone' => $appointment->patient_phone,
        'patient_mobile' => $appointment->patient_mobile,
        'patient_email' => $appointment->patient_email,
        'service_name' => $service_name,
        'doctor_name' => $doctor_name,
        'day' => $appointment->day,
        'time' => $appointment->time,
        'custom_text_1' => $appointment->custom_text_1,
        'custom_text_2' => $appointment->custom_text_2,
        'custom_text_3' => $appointment->custom_text_3,
        'custom_text_4' => $appointment->custom_text_4,
    );

    sab_send_emails($appointment_data, true, $reminder_type);

    $sms_username = get_option('sab_sms_username');
    $sms_password = get_option('sab_sms_password');
    $sms_from = get_option('sab_sms_from');

    if (!empty($sms_username) && !empty($sms_password) && !empty($sms_from)) {
        $sms_user_message = get_option('sab_sms_user_message_' . $reminder_type);

        $sms_use_template_user = get_option('sab_sms_use_template_user_' . $reminder_type, 0);
        $sms_template_id_user = get_option('sab_sms_template_id_user_' . $reminder_type, '');

        $day_name = jdate('l', strtotime($appointment->day));
        $day_number = jdate('j', strtotime($appointment->day));
        $month_name = jdate('F', strtotime($appointment->day));
        $year = jdate('Y', strtotime($appointment->day));
        $formatted_date = "{$day_name} {$day_number} {$month_name} {$year}";

        $replacements = array(
            '{name}' => $appointment->patient_name,
            '{service}' => $service_name,
            '{doctor}' => $doctor_name,
            '{date}' => $formatted_date,
            '{time}' => $appointment->time,
            '{mobile}' => $appointment->patient_mobile,
            '{phone}' => $appointment->patient_phone,
            '{email}' => $appointment->patient_email,
            '{custom-1}' => $appointment->custom_text_1,
            '{custom-2}' => $appointment->custom_text_2,
            '{custom-3}' => $appointment->custom_text_3,
            '{custom-4}' => $appointment->custom_text_4,
        );

        $send_sms = function($to, $message, $use_template, $template_id) use ($sms_username, $sms_password, $sms_from) {
            if ($use_template && !empty($template_id)) {
                $data = array(
                    'username' => $sms_username,
                    'password' => $sms_password,
                    'text' => $message,
                    'to' => $to,
                    'bodyId' => $template_id
                );
                $post_data = http_build_query($data);
                $handle = curl_init('https://rest.payamak-panel.com/api/SendSMS/BaseServiceNumber');
                curl_setopt($handle, CURLOPT_HTTPHEADER, array('content-type' => 'application/x-www-form-urlencoded'));
                curl_setopt($handle, CURLOPT_RETURNTRANSFER, true);
                curl_setopt($handle, CURLOPT_SSL_VERIFYHOST, false);
                curl_setopt($handle, CURLOPT_SSL_VERIFYPEER, false);
                curl_setopt($handle, CURLOPT_POST, true);
                curl_setopt($handle, CURLOPT_POSTFIELDS, $post_data);
                $response = curl_exec($handle);
                error_log('Template SMS Response: ' . print_r($response, true));
            } else {
                $data = array(
                    'username' => $sms_username,
                    'password' => $sms_password,
                    'to' => $to,
                    'from' => $sms_from,
                    'text' => $message
                );
                $post_data = http_build_query($data);
                $handle = curl_init('https://rest.payamak-panel.com/api/SendSMS/SendSMS');
                curl_setopt($handle, CURLOPT_HTTPHEADER, array('content-type' => 'application/x-www-form-urlencoded'));
                curl_setopt($handle, CURLOPT_RETURNTRANSFER, true);
                curl_setopt($handle, CURLOPT_SSL_VERIFYHOST, false);
                curl_setopt($handle, CURLOPT_SSL_VERIFYPEER, false);
                curl_setopt($handle, CURLOPT_POST, true);
                curl_setopt($handle, CURLOPT_POSTFIELDS, $post_data);
                $response = curl_exec($handle);
                curl_close($handle);
                error_log('Regular SMS Sent Response: ' . print_r($response, true));
            }
        };

        $user_message = str_replace(array_keys($replacements), array_values($replacements), $sms_user_message);
        $send_sms($appointment->patient_mobile, $user_message, $sms_use_template_user, $sms_template_id_user);
    }

    $ippanel_username = get_option('sab_ippanel_username');
    $ippanel_password = get_option('sab_ippanel_password');
    $ippanel_from = get_option('sab_ippanel_from');

    if (!empty($ippanel_username) && !empty($ippanel_password) && !empty($ippanel_from)) {
        $ippanel_user_message = get_option('sab_ippanel_user_message_' . $reminder_type);

        $day_name = jdate('l', strtotime($appointment->day));
        $day_number = jdate('j', strtotime($appointment->day));
        $month_name = jdate('F', strtotime($appointment->day));
        $year = jdate('Y', strtotime($appointment->day));
        $formatted_date = "{$day_name} {$day_number} {$month_name} {$year}";

        $replacements = array(
            '{name}' => $appointment->patient_name,
            '{service}' => $service_name,
            '{doctor}' => $doctor_name,
            '{date}' => $formatted_date,
            '{time}' => $appointment->time,
            '{mobile}' => $appointment->patient_mobile,
            '{phone}' => $appointment->patient_phone,
            '{email}' => $appointment->patient_email,
            '{custom-1}' => $appointment->custom_text_1,
            '{custom-2}' => $appointment->custom_text_2,
            '{custom-3}' => $appointment->custom_text_3,
            '{custom-4}' => $appointment->custom_text_4,
        );

        $url = "https://ippanel.com/services.jspd";

        $user_message = str_replace(array_keys($replacements), array_values($replacements), $ippanel_user_message);
        $rcpt_nm_user = array($appointment->patient_mobile);
        $param_user = array(
            'uname' => $ippanel_username,
            'pass' => $ippanel_password,
            'from' => $ippanel_from,
            'message' => $user_message,
            'to' => json_encode($rcpt_nm_user),
            'op' => 'send'
        );
        $handler_user = curl_init($url);
        curl_setopt($handler_user, CURLOPT_CUSTOMREQUEST, "POST");
        curl_setopt($handler_user, CURLOPT_POSTFIELDS, $param_user);
        curl_setopt($handler_user, CURLOPT_RETURNTRANSFER, true);
        curl_exec($handler_user);
    }
}

add_action('sab_24h_reminder_event', function($appointment_id) {
    sab_send_reminders($appointment_id, '24h');
});

add_action('sab_2h_reminder_event', function($appointment_id) {
    sab_send_reminders($appointment_id, '2h');
});
?>